function [ Sed_tr, Sed_dep  ] = SS_transport(ReachData,  Network, extdata, damdata, Fi_r)
%% DAMSE
%
% INPUT :
%
% ReachData      = Struct defining the features of the network reaches
% Network        = 1x1 struct containing for each node info on upstream and downstream nodes
% extdata        = (Px5) vector containing., for the P external sediment flows, the 1) input reach ID; 2) sediment flux delivered [Kg/s], 3-5) D16, D50 and D84 of the input flow;
%                = NxC matrix reporting the grain size frequency of the N reaches for the C sediment classes.
% damdata        = (M x N+1) matrix containing for each dam in the network the node ID and the trapping efficiency for the N sediment classes


%
%----
% OUTPUT: 
%
% Sed_tr          = total sediment flux on each reach
%
% Sed_dep         = deposited sediment on each reach. 
%  


%% Variables extraction from ReachData 

    %reach properties
    Q = [ReachData.Q]';
    n_Man = [ReachData.n]';
    Wac = [ReachData.Wac]';
    Slope = [ReachData.Slope]';
    D16 = [ReachData.D16]';
    D50 = [ReachData.D50]';
    D84 = [ReachData.D84]';
    tr_limit = [ReachData.tr_limit]';

    NH = Network.NH ;

    %ReachData is extracted from the function "extract_river_network" from
    %CASCADE model by Schmitt et al., 2016 WRR
    


%% sediment boundary 


    sed_contribution(:,1) = [extdata.reach_id];
    sed_contribution(:,2:end) = GSDcurvefit ([extdata.D16]', [extdata.D50]' , [extdata.D84]').*[extdata.sed_flow]';
 
    
%% dams definition
    DamID = damdata(:,1);
    trap = damdata(:,2);
    [uniqueID, idx1, idx2] = unique(DamID);
    countsID = accumarray(idx2(:), 1);
    Reach_Trap = zeros(length(idx2),1);
    count = 1;

    for i = 1:length(uniqueID)
        Reach_Trap(count:count+countsID(i)-1) = mean(trap(count:count+countsID(i)-1));
        count = count + countsID(i);
    end

    uniqueTrap = Reach_Trap(idx1);

    damdata_class = zeros(length(damstruct),length(psi)+1);
    damdata_class(:,1) = [damstruct.node_id];
    damdata_class(:, sed_list(1).psi_class +1) = repmat([damstruct.cobble_trap]',1,length( sed_list(1).psi_class) );
    damdata_class(:, sed_list(2).psi_class +1) = repmat([damstruct.gravel_trap]',1,length( sed_list(2).psi_class) );
    damdata_class(:, sed_list(3).psi_class +1) = repmat([damstruct.sand_trap]',1,length( sed_list(3).psi_class) );
    damdata_class(:, sed_list(4).psi_class +1) = repmat([damstruct.silt_trap]',1,length( sed_list(4).psi_class) );
   
    
    %% sources reaches identification

    %find source reaches in ReachData
    sources = find (cellfun(@isempty,Network.Upstream.Node)==1);
    
    %% Using Zhang et al equation to calculation transport capacity
    %% references

    % Zhang, R. Sediment Hydraulics.  (China Water&Power Press, 1998).

    %% variables initialization
   

    dmi = 2.^(-psi)./1000; %sediment classes diameter (m)

    rho_w = 1000; %water density
    rho_s = 2650; %sediment density
    g = 9.81; %gravity acceleration

    % settling velocity of grain D50
    VS = 1.24; %viscosity coefficient

    SV_D16 = ((13.95*VS./D16)^2+1.06*(rho_s-rho_w)/rho_w*g.*D16).*0.5-13.95*VS./D16;
    SV_D50 = ((13.95*VS./D50)^2+1.06*(rho_s-rho_w)/rho_w*g.*D50).*0.5-13.95*VS./D50;
    SV_D84 = ((13.95*VS./D84)^2+1.06*(rho_s-rho_w)/rho_w*g.*D84).*0.5-13.95*VS./D84;

    SV = 0.16.*SV_D16 + 0.34.*SV_D50 + 0.16.*SV_D84;
    %SV: settling velocity

    QS_Zhang = 0.163.* (U^3 ./G ./ R ./ SV)^0.92; %U is flow velocity; R is hydraulic radius.Both can be obtained from hydraulic solver


    %% Transport capacity 
 
    % dimensionless transport
    G_gr = QS_Zhang;

    % weight concentration of bed material (Kg_sed / Kg_water)
    QS_ppm = G_gr * (R + 1) * D_AW * (v/u_ast)^n / h;

    % transport capacity (Kg_sed / s)
    QS_AW = rho_w * Q * QS_ppm ;

    tr_cap = QS_AW;
    %% Routing scheme

   for n = NH
        
        if any(n == sources) %check if the node is a source node
            
            %choose the transport capacity equation to be used and apply supply limitation
            
            Qbi_tr(n,n,:) = tr_cap_junction( indx_tr_cap, indx_partition, Fi_r(n,:) , D50(n), Slope(n), Q(n), Wac(n), v(n) , h(n) );
            Qbi_tr(n,n,:) = min(squeeze(Qbi_tr(n,n,:))', squeeze(Qbi_tr(n,n,:))'.*tr_limit(n) + nansum( sed_contribution(sed_contribution(:,1) == n,2:end),1));
            
            Qbi_dep(n,n,:) = max(nansum( sed_contribution(sed_contribution(:,1) == n,2:end),1) - squeeze(Qbi_tr(n,n,:))',0);
                  
        else
            
            %check for incoming sediment flows from the river network
            Qbi_incoming = squeeze(nansum(Qbi_tr(:,Network.Upstream.Node{n},:),2));
           
            %check if there are external sediment flows
            Qbi_external = nansum( sed_contribution(sed_contribution(:,1) == n,2:end),1);    
            Qbi_incoming(n,:) = Qbi_external; 
            
            if any(any(Qbi_incoming))  % check if there are incoming sediment cascades 
                
                [id_sel, ~] = find(QB_tr(:,Network.Upstream.Node{n})>0);  %id of the nodes of the incoming cascades
                
                Qbi_total = nansum(Qbi_incoming); %total incoming sediment flux for classes
                
                % Find in which percentage the flows of the sub-cascades contribute to the total incoming flow of the sed. class
                % (Required to divide the eventual deposited sediment load proportionally between cascades)
                Qbi_perc = Qbi_incoming./ Qbi_total; 
                Qbi_perc(isnan(Qbi_perc)) = 0;           
                
                % Choose the transport capacity equation to be used
                tr_cap = tr_cap_junction( indx_tr_cap , indx_partition , Fi_r(n,:) , D50(n),Slope(n), Q(n), Wac(n),v(n) , h(n) );
            
                % part_tr >0 Deposit, <0 Eroded
                part_tr = Qbi_total - tr_cap; 
                
                Qbi_tr(id_sel,n,:) = zeros(size(Qbi_tr(id_sel,n,:)));
                Qbi_dep([id_sel;n],n,:) = zeros(size(Qbi_dep([id_sel; n],n,:)));
                
                % part_tr >0 means deposit
                % The relative sub-cascade in the consiedered node is not
                % activated, while all the other sub-cascades deposit
                
                if ~isempty(part_tr(part_tr>0))
                    
                    %all the incoming sub cascades deposit part of their loads, according to Qbi_perc
                    Qbi_dep([id_sel;n],n,part_tr>0) = part_tr(part_tr>0).* Qbi_perc ([id_sel;n],part_tr>0);
                    
                    %the remaining sediment loads proceed downstream
                    Qbi_tr(id_sel,n,part_tr>0) = max(nansum(Qbi_tr(id_sel,Network.Upstream.Node{n},part_tr>0),2) -  Qbi_dep(id_sel,n,part_tr>0),0);

                    %deposit the relative fraction of external sed.flow 
                    Qbi_external(part_tr>0) = Qbi_external(part_tr>0) - squeeze(Qbi_dep(n,n,part_tr>0))';
                   
                    %for the sed. classes that exceed the transport
                    %capacity, no new sub-cascade is created, except when
                    %external sed.flows are present.
                    Qbi_tr(n,n,part_tr>0) = Qbi_external(part_tr>0) ;
                    
                end

                % part_tr <0 means the incoming CASCADE is not enough to satisfy the transport capacity of the reach
                % the analysed reach is activated for that specific grain size class
                
                if ~isempty(part_tr(part_tr<0))
                    
                    %quantify the entrained sediment flow by measuring the exceeding transport capacity, 
                    %applying supply limitation via tr_limit and adding the external sediment contributions
                    Qbi_tr(n,n,part_tr<0) = Qbi_external(part_tr<0) - part_tr(part_tr<0) .* tr_limit(n);
                    
                    % all the incoming sub_cascades for the sediment classes with part_tr <0 proceed downstream unaltered
                    Qbi_tr(id_sel,n,part_tr<0) = nansum(Qbi_tr(id_sel,Network.Upstream.Node{n},part_tr<0),2);
 
                end
                
            else %if there are no incoming sediment cascades, measure the trasport capacity of the reach and create a new cascade accordingly
                
                %choose the transport capacity equation to be used+                              
                Qbi_tr(n,n,:) = tr_limit(n) .* tr_cap_junction( indx_tr_cap , indx_partition , Fi_r(n,:) , D50(n),Slope(n), Q(n), Wac(n),v(n) , h(n) );
                               
            end
            
            
        end 
        
        %chech if there is a dam on the To-Node of the reach
        if isempty(damdata_class(damdata_class(:,1) == ReachData(n).FromN ,2:end))          
            trap_efficiency = zeros(1,length(psi));
        else
            trap_efficiency = damdata_class(damdata_class(:,1)== ReachData(n).FromN  ,2:end);
        end

        %apply effect of dams
        Qbi_dep(:,n,:) = squeeze(Qbi_dep(:,n,:)) + squeeze(Qbi_tr(:,n,:)) .* trap_efficiency;
        Qbi_tr(:,n,:) = squeeze(Qbi_tr(:,n,:)) .* (1-trap_efficiency);
   
        % Update total flows transported and deposited
        QB_tr(:,n)= nansum(Qbi_tr(:,n,:),3);
        Sed_dep(:,n)= nansum(Qbi_dep(:,n,:),3);
        

    end
    
    
end   